<?php
/**
 * Settings class file.
 *
 * @package WordPress Plugin Template/Settings
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Settings class.
 */
class All_in_One_Video_Downloader_Settings
{

    /**
     * The single instance of All_in_One_Video_Downloader_Settings.
     *
     * @var     object
     * @access  private
     * @since   1.0.0
     */
    private static $_instance = null; //phpcs:ignore

    /**
     * The main plugin object.
     *
     * @var     object
     * @access  public
     * @since   1.0.0
     */
    public $parent = null;

    /**
     * Prefix for plugin settings.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $base = '';

    /**
     * Available settings for plugin.
     *
     * @var     array
     * @access  public
     * @since   1.0.0
     */
    public $settings = array();

    /**
     * Constructor function.
     *
     * @param object $parent Parent object.
     */
    public function __construct($parent)
    {
        $this->parent = $parent;

        $this->base = 'aiodl_';

        // Initialise settings.
        add_action('init', array($this, 'init_settings'), 11);

        // Register plugin settings.
        add_action('admin_init', array($this, 'register_settings'));

        // Add settings page to menu.
        add_action('admin_menu', array($this, 'add_menu_item'));

        // Add settings link to plugins page.
        add_filter(
            'plugin_action_links_' . plugin_basename($this->parent->file),
            array(
                $this,
                'add_settings_link',
            )
        );

        // Configure placement of plugin settings page. See readme for implementation.
        add_filter($this->base . 'menu_settings', array($this, 'configure_settings'));
    }

    /**
     * Initialise settings
     *
     * @return void
     */
    public function init_settings()
    {
        $this->settings = $this->settings_fields();
    }

    /**
     * Add settings page to admin menu
     *
     * @return void
     */
    public function add_menu_item()
    {

        $args = $this->menu_settings();

        // Do nothing if wrong location key is set.
        if (is_array($args) && isset($args['location']) && function_exists('add_' . $args['location'] . '_page')) {
            switch ($args['location']) {
                case 'options':
                case 'submenu':
                    $page = add_submenu_page($args['parent_slug'], $args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function']);
                    break;
                case 'menu':
                    $page = add_menu_page($args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function'], $args['icon_url'], $args['position']);
                    break;
                default:
                    return;
            }
            add_action('admin_print_styles-' . $page, array($this, 'settings_assets'));
        }
    }

    /**
     * Prepare default settings page arguments
     *
     * @return mixed|void
     */
    private function menu_settings()
    {
        return apply_filters(
            $this->base . 'menu_settings',
            array(
                'location' => 'menu', // Possible settings: options, menu, submenu.
                'parent_slug' => 'options-general.php',
                'page_title' => __('Video Downloader Settings', 'all-in-one-video-downloader'),
                'menu_title' => __('Video Downloader Settings', 'all-in-one-video-downloader'),
                'capability' => 'manage_options',
                'menu_slug' => $this->parent->_token . '_settings',
                'function' => array($this, 'settings_page'),
                'icon_url' => '',
                'position' => null,
            )
        );
    }

    /**
     * Container for settings page arguments
     *
     * @param array $settings Settings array.
     *
     * @return array
     */
    public function configure_settings($settings = array())
    {
        return $settings;
    }

    /**
     * Load settings JS & CSS
     *
     * @return void
     */
    public function settings_assets()
    {

        // We're including the farbtastic script & styles here because they're needed for the colour picker
        // If you're not including a colour picker field then you can leave these calls out as well as the farbtastic dependency for the wpt-admin-js script below.
        wp_enqueue_style('farbtastic');
        wp_enqueue_script('farbtastic');

        // We're including the WP media scripts here because they're needed for the image upload field.
        // If you're not including an image upload then you can leave this function call out.
        wp_enqueue_media();

        wp_register_script($this->parent->_token . '-settings-js', $this->parent->assets_url . 'js/settings' . $this->parent->script_suffix . '.js', array('farbtastic', 'jquery'), '1.0.0', true);
        wp_enqueue_script($this->parent->_token . '-settings-js');
    }

    /**
     * Add settings link to plugin list table
     *
     * @param array $links Existing links.
     * @return array        Modified links.
     */
    public function add_settings_link($links)
    {
        $settings_link = '<a href="options-general.php?page=' . $this->parent->_token . '_settings">' . __('Settings', 'all-in-one-video-downloader') . '</a>';
        array_push($links, $settings_link);
        return $links;
    }

    /**
     * Build settings fields
     *
     * @return array Fields to be displayed on settings page
     */
    private function settings_fields()
    {
        $settings['general'] = array(
            'title' => __('General', 'all-in-one-video-downloader'),
            'description' => __('General settings about the plugin.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'show_mp3',
                    'label' => __('Show M4A as MP3', 'all-in-one-video-downloader'),
                    'description' => __('If you enable this YouTube videos will be shown with MP3 extension also.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'hide_dash',
                    'label' => __('Hide dash videos', 'all-in-one-video-downloader'),
                    'description' => __('If you enable this YouTube videos without sound will be hidden.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'disable_session',
                    'label' => __('Disable session storage', 'all-in-one-video-downloader'),
                    'description' => __('Enable this option only if you see blank page after clicking the video links.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'bandwidth_saving_mode',
                    'label' => __('Bandwidth saving mode', 'all-in-one-video-downloader'),
                    'description' => __('If you enable this download links will be redirected to video URLs. Downloads may not start by automatically.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'recaptcha',
                    'label' => __('Enable Recaptcha', 'all-in-one-video-downloader'),
                    'description' => __('Before enabling this be sure you have added Google Recaptcha api keys on Api Keys tab.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'enable_latest_downloads',
                    'label' => __('Latest downloads', 'all-in-one-video-downloader'),
                    'description' => __('If you enable this latest downloaded videos will be recorded in the database.', 'all-in-one-video-downloader'),
                    'type' => 'checkbox',
                    'default' => '',
                ),
                /*
                array(
                    'id' => 'latest_downloads_count',
                    'label' => __('Latest downloads count', 'all-in-one-video-downloader'),
                    'description' => __('By default last 10 downloads will be recorded.', 'all-in-one-video-downloader'),
                    'type' => 'number',
                    'default' => '10',
                    'placeholder' => __('Enter an integer', 'all-in-one-video-downloader'),
                ),*/
                array(
                    'id' => 'download_timer',
                    'label' => __('Download timer', 'all-in-one-video-downloader'),
                    'description' => __('Users will wait to start download after clicking a download link. Set it zero to disable waiting.', 'all-in-one-video-downloader'),
                    'type' => 'number',
                    'default' => '0',
                    'placeholder' => __('Enter an integer', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'filename_suffix',
                    'label' => __('Suffix for filenames', 'all-in-one-video-downloader'),
                    'description' => __('This text will be appended after video title.', 'all-in-one-video-downloader'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Enter a text', 'all-in-one-video-downloader'),
                ),
            ),
        );
        $settings['api'] = array(
            'title' => __('Api Keys', 'all-in-one-video-downloader'),
            'description' => __('Enter your api keys here.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'facebook_cookies',
                    'label' => __('Facebook cookies', 'all-in-one-video-downloader'),
                    'description' => __('<a href="https://support.nicheoffice.web.tr/help-center/articles/1/22/how-to-get-facebook-cookies" target="_blank">Click to learn how to find.</a>', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your cookies', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'snapchat_cookies',
                    'label' => 'Snapchat cookies',
                    'description' => '<a href="https://support.nicheoffice.web.tr/help-center/articles/1/10/26/how-to-get-snapchat-cookies" target="_blank">Click to learn how to find.</a>',
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your cookies', 'all-in-one-video-downloader'),
                ),
                /*
                array(
                    'id' => 'instagram_cookies',
                    'label' => __('Instagram cookies', 'all-in-one-video-downloader'),
                    'description' => __('<a href="https://support.nicheoffice.web.tr/help-center/articles/1/10/15/how-to-get-instagram-cookies" target="_blank">Click to learn how to find.</a>', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your cookies', 'all-in-one-video-downloader'),
                ),*/
                array(
                    'id' => 'soundcloud_api_key',
                    'label' => __('Soundcloud api key', 'all-in-one-video-downloader'),
                    'description' => __('<a href="https://support.nicheoffice.web.tr/help-center/articles/12/how-to-get-a-soundcloud-api-key" target="_blank">Click to learn how can you get it.</a>', 'all-in-one-video-downloader'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Paste your api key', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'rapid_api_key',
                    'label' => __('Rapid api key', 'all-in-one-video-downloader'),
                    'description' => '<a href="https://support.nicheoffice.web.tr/help-center/articles/27/how-to-get-rapidapi-api-key" target="_blank">Click to learn how can you get it.</a>',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => 'Paste your api key',
                ),
                array(
                    'id' => 'recaptcha_public_api_key',
                    'label' => __('Recaptcha V3 public key', 'all-in-one-video-downloader'),
                    'description' => __('<a href="https://www.google.com/recaptcha/admin/create" target="_blank">Click to get an api key.</a>', 'all-in-one-video-downloader'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Paste your api key', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'recaptcha_private_api_key',
                    'label' => __('Recaptcha V3 private key', 'all-in-one-video-downloader'),
                    'description' => __('<a href="https://www.google.com/recaptcha/admin/create" target="_blank">Click to get an api key.</a>', 'all-in-one-video-downloader'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Paste your api key', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'tracking_code',
                    'label' => __('Tracking code', 'all-in-one-video-downloader'),
                    'description' => __('It will be inserted inside footer.', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your Javascript tracking code <script>...</script>', 'all-in-one-video-downloader'),
                ),
            ),
        );
        $settings['ads'] = array(
            'title' => __('Advertising', 'all-in-one-video-downloader'),
            'description' => __('You can paste your ad codes here.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'ad_area_1',
                    'label' => __('Ad Area 1', 'all-in-one-video-downloader'),
                    'description' => __('It will be placed under the download form.', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your ad code', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'ad_area_2',
                    'label' => __('Ad Area 2', 'all-in-one-video-downloader'),
                    'description' => __('It will be placed before video title.', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your ad code', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'ad_area_3',
                    'label' => __('Ad Area 3', 'all-in-one-video-downloader'),
                    'description' => __('It will be placed top of articles and pages.', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your ad code', 'all-in-one-video-downloader'),
                ),
                array(
                    'id' => 'ad_area_4',
                    'label' => __('Ad Area 4', 'all-in-one-video-downloader'),
                    'description' => __('It will be placed under articles and pages.', 'all-in-one-video-downloader'),
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => __('Paste your ad code', 'all-in-one-video-downloader'),
                ),
            ),
        );
        $settings['monetization'] = array(
            'title' => 'Monetization',
            'description' => 'You can monetize video downloads by redirecting to paid ad platforms.',
            'fields' => array(
                array(
                    'id' => 'url_monetization',
                    'label' => 'Enable URL shortener',
                    'description' => 'Enable this option if you want to make downloads start after skipping ads.',
                    'type' => 'checkbox',
                    'default' => '',
                    'placeholder' => '',
                ),
                array(
                    'id' => 'url_shortener',
                    'label' => 'URL shortener service',
                    'description' => 'Choose the service which you want to use to shorten the urls.',
                    'type' => 'select',
                    'options' => array(
                        //'shortest' => 'Shorte.st',
                        'bcvc' => 'Bc.vc',
                    ),
                ),
                array(
                    'id' => 'shortest_api_key',
                    'label' => 'Shorte.st api key',
                    'description' => '',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => 'Paste your api key here',
                ),
                array(
                    'id' => 'bcvc_api_key',
                    'label' => 'Bc.vc api key',
                    'description' => '',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => 'Paste your api key here',
                ),
                array(
                    'id' => 'bcvc_user_id',
                    'label' => 'Bc.vc user id',
                    'description' => '',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => 'Paste your user id here',
                ),
            ),
        );
        $settings['appearance'] = array(
            'title' => __('Appearance', 'all-in-one-video-downloader'),
            'description' => __('You can modify your website look here.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'show_howto_download',
                    'label' => __('Show how to download section', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'show_features',
                    'label' => __('Show features section', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'show_supported',
                    'label' => __('Show supported sources section', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'show_blog',
                    'label' => __('Show blog posts at the homepage', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'show_social_media',
                    'label' => __('Show social media buttons', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'show_header_image',
                    'label' => 'Show header image next to downloader form',
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                    'placeholder' => 'Enter logo height in px',
                ),
                array(
                    'id' => 'logo_height',
                    'label' => 'Logo Height',
                    'description' => '',
                    'type' => 'number',
                    'default' => '28',
                    'placeholder' => 'Enter logo height in px',
                ),
                /*
                array(
                    'id' => 'show_logo',
                    'label' => __('Show logo in header', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox',
                    'default' => '',
                ),
                array(
                    'id' => 'logo_image',
                    'label' => __('Logo image', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'image',
                    'default' => '',
                    'placeholder' => '',
                ),*/
                array(
                    'id' => 'facebook_username',
                    'label' => __('Facebook username', 'all-in-one-video-downloader'),
                    'description' => '',
                    'placeholder' => 'Facebook username',
                    'type' => 'text',
                    'default' => '',
                ),
                array(
                    'id' => 'twitter_username',
                    'label' => __('Twitter username', 'all-in-one-video-downloader'),
                    'description' => '',
                    'placeholder' => 'Twitter username',
                    'type' => 'text',
                    'default' => '',
                ),
                array(
                    'id' => 'youtube_username',
                    'label' => __('YouTube username', 'all-in-one-video-downloader'),
                    'description' => '',
                    'placeholder' => 'YouTube username',
                    'type' => 'text',
                    'default' => '',
                ),
                array(
                    'id' => 'instagram_username',
                    'label' => __('Instagram username', 'all-in-one-video-downloader'),
                    'description' => '',
                    'placeholder' => 'Instagram username',
                    'type' => 'text',
                    'default' => '',
                ),
                array(
                    'id' => 'homepage_content',
                    'label' => 'Custom HTML code to insert to homepage',
                    'description' => '',
                    'placeholder' => '',
                    'type' => 'editor',
                    'default' => '',
                ),
            ),
        );
        $settings['downloaders'] = array(
            'title' => __('Downloaders', 'all-in-one-video-downloader'),
            'description' => __('Choose which downloaders can be used on your website.', 'all-in-one-video-downloader'),
            'fields' => array(),
        );
        $downloaders = All_in_One_Video_Downloader_Downloaders::$downloaders;
        foreach ($downloaders as $downloader) {
            $settings['downloaders']['fields'][] = array(
                'id' => 'downloader_' . $downloader['slug'],
                'label' => __('Enable ' . $downloader['name'] . ' downloader', 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'checkbox',
                'default' => 'on',
            );
            $settings['downloaders']['fields'][] = array(
                'id' => 'proxy_' . $downloader['slug'],
                'label' => __('Enable proxies for ' . $downloader['name'], 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'checkbox',
                'default' => '',
            );
            /*
            array_push($settings['downloaders']['fields'], array(
                'id' => 'downloader_text_' . $downloader['slug'],
                'label' => __('Downloader text for ' . $downloader['name'], 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'text',
                'default' => '',
                'placeholder' => ''
            ));
            array_push($settings['downloaders']['fields'], array(
                    'id' => 'color_' . $downloader['slug'],
                    'label' => __('Button color for ' . $downloader['name'], 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'color',
                    'default' => $downloader['color'],
                )
            );
            */
            $settings['downloaders']['fields'][] = array(
                'id' => 'downloader_slug_' . $downloader['slug'],
                'label' => __('Page slug for ' . $downloader['name'], 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'text',
                'default' => '',
                'placeholder' => ''
            );
        }
        $settings['proxy'] = array(
            'title' => __('Proxy', 'all-in-one-video-downloader'),
            'description' => __('Settings about proxies. You can leave blank username/password field if they are not required.', 'all-in-one-video-downloader'),
            'hide' => true,
            'fields' => array(
                array(
                    'id' => 'proxy_count',
                    'label' => __('Number of proxies', 'all-in-one-video-downloader'),
                    'description' => __('Enter how many proxies will you use.', 'all-in-one-video-downloader'),
                    'type' => 'number',
                    'default' => '0',
                    'placeholder' => __('Enter an integer', 'all-in-one-video-downloader'),
                ),
            ),
        );
        for ($i = 0; $i < get_option('aiodl_proxy_count'); $i++) {
            $settings['proxy']['fields'][] = array(
                'id' => 'proxy_ip_' . $i,
                'label' => __('Proxy ip', 'all-in-one-video-downloader'),
                'description' => 'Details for proxy ' . ($i + 1),
                'type' => 'text',
                'default' => '',
                'placeholder' => __('Enter an IP address', 'all-in-one-video-downloader'),
            );
            $settings['proxy']['fields'][] = array(
                'id' => 'proxy_port_' . $i,
                'label' => __('Proxy port', 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'text',
                'default' => '',
                'placeholder' => __('Enter port number', 'all-in-one-video-downloader'),
            );
            $settings['proxy']['fields'][] = array(
                'id' => 'proxy_username_' . $i,
                'label' => __('Proxy username', 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'text',
                'default' => '',
                'placeholder' => __('Enter username', 'all-in-one-video-downloader'),
            );
            $settings['proxy']['fields'][] = array(
                'id' => 'proxy_password_' . $i,
                'label' => __('Proxy password', 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'text',
                'default' => '',
                'placeholder' => __('Enter password', 'all-in-one-video-downloader'),
            );
            $settings['proxy']['fields'][] = array(
                'id' => 'proxy_type_' . $i,
                'label' => __('Proxy type', 'all-in-one-video-downloader'),
                'description' => '',
                'type' => 'select',
                'options' => array(
                    'http' => 'HTTP',
                    'https' => 'HTTPS',
                    'socks4' => 'Socks4',
                    'socks5' => 'Socks5',
                ),
                'default' => 'http',
            );
        }
        $settings['rest-api'] = array(
            'title' => __('Rest API', 'all-in-one-video-downloader'),
            'description' => __('You can use the rest API on your other apps.<br>Your REST api endpoint is: ' . get_site_url() . '/wp-json/aio-dl/api/', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'rest_api',
                    'label' => __('Enable REST Api', 'all-in-one-video-downloader'),
                    'description' => '<a href="https://support.nicheoffice.web.tr/help-center/articles/1/23/how-to-enable-and-use-rest-api" target="_blank">Click to learn how to use</a>',
                    'type' => 'checkbox',
                    'default' => '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'rest_api_ip_check',
                    'label' => __('Enable IP protection', 'all-in-one-video-downloader'),
                    'description' => 'Check this option if you want to give access to REST api for specific IP addresses',
                    'type' => 'checkbox',
                    'default' => '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'rest_api_key',
                    'label' => __('Api key', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'text',
                    'default' => $this->getRestApiKey(),
                    'placeholder' => ''
                ),
                array(
                    'id' => 'rest_api_allowed_ips',
                    'label' => 'Allowed IPs',
                    'description' => 'Enter one IP each line',
                    'type' => 'textarea',
                    'default' => '',
                    'placeholder' => ''
                ),
            ),
        );
        $settings['latest-downloads'] = array(
            'title' => __('Latest Downloads', 'all-in-one-video-downloader'),
            'description' => 'You can see latest downloaded items here. If you do not see please enable latest downloads feature on general settings.',
            'fields' => array(
                array(
                    'id' => 'latest_downloads',
                    'label' => '',
                    'description' => '',
                    'type' => 'latest_downloads',
                    'default' => '',
                    'placeholder' => ''
                ),
            ),
        );
        $settings['license'] = array(
            'title' => __('License', 'all-in-one-video-downloader'),
            'description' => __('License settings.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'license_name',
                    'label' => __('Your name', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_email',
                    'label' => __('Your email', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'email',
                    'default' => '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_code',
                    'label' => __('Purchase code', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_domain',
                    'label' => 'Website URL',
                    'description' => '',
                    'type' => 'text_disabled_none',
                    'default' => get_site_url(),
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_version',
                    'label' => 'Plugin version',
                    'description' => '',
                    'type' => 'text_disabled_none',
                    'default' => AIO_VIDEO_DOWNLOADER_VERSION,
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_registration',
                    'label' => 'Registration code',
                    'description' => 'Enter your name, email and purchase code and save settings to generate the registration code.',
                    'type' => 'text_disabled_none',
                    'default' => $this->createRegistrationCode(),
                    'placeholder' => ''
                ),
                array(
                    'id' => 'license_fingerprint',
                    'label' => 'Activation code',
                    'description' => '<a href="https://nicheoffice.web.tr/activate" target="_blank">Click to get it.</a>',
                    'type' => 'text_reverse',
                    'default' => '',
                    'placeholder' => ''
                ),
            ),
        );
        $settings['system-check'] = array(
            'title' => __('System Check', 'all-in-one-video-downloader'),
            'description' => __('Check system requirements.', 'all-in-one-video-downloader'),
            'fields' => array(
                array(
                    'id' => 'php_version',
                    'label' => __('PHP version 7.0 or higher', 'all-in-one-video-downloader'),
                    'description' => 'PHP Version: ' . PHP_VERSION,
                    'type' => 'checkbox_system',
                    'default' => '',
                    'condition' => explode('.', PHP_VERSION)[0] >= 7,
                    'placeholder' => '',
                ),
                array(
                    'id' => 'curl_enabled',
                    'label' => __('cURL installed', 'all-in-one-video-downloader'),
                    'description' => 'cURL Version: ' . (isset(curl_version()["version"]) != '' ? curl_version()["version"] : ''),
                    'type' => 'checkbox_system',
                    'default' => '',
                    'condition' => isset(curl_version()["version"]) != '',
                    'placeholder' => ''
                ),
                array(
                    'id' => 'mbstring_enabled',
                    'label' => __('mbstring installed', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox_system',
                    'default' => '',
                    'condition' => extension_loaded('mbstring'),
                    'placeholder' => ''
                ),
                array(
                    'id' => 'allowurlfopen_enabled',
                    'label' => __('allow_url_fopen enabled', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox_system',
                    'default' => '',
                    'condition' => ini_get('allow_url_fopen'),
                    'placeholder' => ''
                ),
                /*
                array(
                    'id' => 'modrewrite_enabled',
                    'label' => __('mod_rewrite installed', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'checkbox_system',
                    'default' => '',
                    'condition' => $this->checkModRewrite(),
                    'placeholder' => ''
                ),*/
                array(
                    'id' => 'server_software',
                    'label' => __('Server software', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'info_text',
                    'default' => $_SERVER["SERVER_SOFTWARE"],
                    'placeholder' => ''
                ),
                array(
                    'id' => 'server_ip',
                    'label' => __('Server IP', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'info_text',
                    'default' => $_SERVER["SERVER_ADDR"],
                    'placeholder' => ''
                ),
                array(
                    'id' => 'server_os',
                    'label' => __('Server OS', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'info_text',
                    'default' => php_uname(),
                    'placeholder' => ''
                ),
                array(
                    'id' => 'cache_size',
                    'label' => __('Cache Size', 'all-in-one-video-downloader'),
                    'description' => '',
                    'type' => 'info_text',
                    'default' => $this->cacheSize(__DIR__ . '/../cache/'),
                    'placeholder' => ''
                ),
            ),
        );
        $settings = apply_filters($this->parent->_token . '_settings_fields', $settings);
        return $settings;
    }

    public function createRegistrationCode()
    {
        $code = [];
        $nameValid = !empty(get_option('aiodl_license_name'));
        $emailValid = !empty(get_option('aiodl_license_email'));
        $codeValid = !empty(get_option('aiodl_license_code'));
        $ipValid = filter_var($_SERVER['SERVER_ADDR'], FILTER_VALIDATE_IP);
        $versionValid = !empty(AIO_VIDEO_DOWNLOADER_VERSION);
        if ($nameValid && $emailValid && $codeValid && $ipValid && $versionValid) {
            $code['name'] = get_option('aiodl_license_name');
            $code['email'] = get_option('aiodl_license_email');
            $code['code'] = get_option('aiodl_license_code');
            $code['url'] = get_site_url();
            $code['ip'] = $_SERVER['SERVER_ADDR'];
            $code['version'] = AIO_VIDEO_DOWNLOADER_VERSION;
            $code = serialize($code);
        } else {
            $code = '';
        }
        return strrev(base64_encode($code));
    }

    public function getRestApiKey()
    {
        $apiKey = get_option('aiodl_rest_api_key');
        if ($apiKey == '') {
            return $this->generateToken();
        } else {
            return $apiKey;
        }
    }

    public function generateToken()
    {
        if (defined('PHP_MAJOR_VERSION') && PHP_MAJOR_VERSION > 5) {
            return bin2hex(random_bytes(32));
        } else {
            if (function_exists('mcrypt_create_iv')) {
                return bin2hex(mcrypt_create_iv(32, MCRYPT_DEV_URANDOM));
            } else {
                return bin2hex(openssl_random_pseudo_bytes(32));
            }
        }
    }

    public function cleanCache($dir, $time = 86400)
    {
        foreach (glob($dir . "*") as $file) {
            if (time() - filectime($file) > $time) {
                unlink($file);
            }
        }
    }

    public function cacheSize($path)
    {
        $this->cleanCache($path);
        $bytestotal = 0;
        $path = realpath($path);
        if ($path !== false && $path != '' && file_exists($path)) {
            foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS)) as $object) {
                $bytestotal += $object->getSize();
            }
        }
        require_once __DIR__ . '/Helpers.php';
        return Helpers::formatSize($bytestotal);
    }

    public function checkModRewrite()
    {
        if (function_exists('apache_get_modules')) {
            $modules = apache_get_modules();
            return in_array('mod_rewrite', $modules);
        }
        return false;
    }

    /**
     * Register plugin settings
     *
     * @return void
     */
    public function register_settings()
    {
        if (is_array($this->settings)) {

            // Check posted/selected tab.
            //phpcs:disable
            $current_section = '';
            if (isset($_POST['tab']) && $_POST['tab']) {
                $current_section = $_POST['tab'];
            } else {
                if (isset($_GET['tab']) && $_GET['tab']) {
                    $current_section = $_GET['tab'];
                }
            }
            //phpcs:enable

            foreach ($this->settings as $section => $data) {

                if ($current_section && $current_section !== $section) {
                    continue;
                }

                // Add section to page.
                add_settings_section($section, $data['title'], array($this, 'settings_section'), $this->parent->_token . '_settings');

                foreach ($data['fields'] as $field) {

                    // Validation callback for field.
                    $validation = '';
                    if (isset($field['callback'])) {
                        $validation = $field['callback'];
                    }

                    // Register field.
                    $option_name = $this->base . $field['id'];
                    register_setting($this->parent->_token . '_settings', $option_name, $validation);

                    // Add field to page.
                    add_settings_field(
                        $field['id'],
                        $field['label'],
                        array($this->parent->admin, 'display_field'),
                        $this->parent->_token . '_settings',
                        $section,
                        array(
                            'field' => $field,
                            'prefix' => $this->base,
                        )
                    );
                }

                if (!$current_section) {
                    break;
                }
            }
        }
    }

    /**
     * Settings section.
     *
     * @param array $section Array of section ids.
     * @return void
     */
    public function settings_section($section)
    {
        $html = '<p> ' . $this->settings[$section['id']]['description'] . '</p>' . "\n";
        echo $html; //phpcs:ignore
    }

    /**
     * Load settings page content.
     *
     * @return void
     */
    public function settings_page()
    {

        // Build page HTML.
        $html = '<div class="wrap" id="' . $this->parent->_token . '_settings">' . "\n";
        $html .= '<h2>' . __('Plugin Settings', 'all-in-one-video-downloader') . '</h2>' . "\n";

        $tab = '';
        //phpcs:disable
        if (isset($_GET['tab']) && $_GET['tab']) {
            $tab .= $_GET['tab'];
        }
        //phpcs:enable

        // Show page tabs.
        if (is_array($this->settings) && 1 < count($this->settings)) {

            $html .= '<h2 class="nav-tab-wrapper">' . "\n";

            $c = 0;
            foreach ($this->settings as $section => $data) {

                // Set tab class.
                $class = 'nav-tab';
                if (!isset($_GET['tab'])) { //phpcs:ignore
                    if (0 === $c) {
                        $class .= ' nav-tab-active';
                    }
                } else {
                    if (isset($_GET['tab']) && $section == $_GET['tab']) { //phpcs:ignore
                        $class .= ' nav-tab-active';
                    }
                }

                // Set tab link.
                $tab_link = add_query_arg(array('tab' => $section));
                if (isset($_GET['settings-updated'])) { //phpcs:ignore
                    $tab_link = remove_query_arg('settings-updated', $tab_link);
                }

                // Output tab.
                $html .= '<a href="' . $tab_link . '" class="' . esc_attr($class) . '">' . esc_html($data['title']) . '</a>' . "\n";

                ++$c;
            }

            $html .= '</h2>' . "\n";
        }

        $html .= '<form method="post" action="options.php" enctype="multipart/form-data">' . "\n";

        // Get settings fields.
        ob_start();
        settings_fields($this->parent->_token . '_settings');
        do_settings_sections($this->parent->_token . '_settings');
        $html .= ob_get_clean();

        $html .= '<p class="submit">' . "\n";
        $html .= '<input type="hidden" name="tab" value="' . esc_attr($tab) . '" />' . "\n";
        $html .= '<input name="Submit" type="submit" class="button-primary" value="' . esc_attr(__('Save Settings', 'all-in-one-video-downloader')) . '" />' . "\n";
        $html .= '</p>' . "\n";
        $html .= '</form>' . "\n";
        $html .= '</div>' . "\n";
        self::saveInstagramCookie();
        echo $html; //phpcs:ignore
    }

    public static function saveInstagramCookie()
    {
        $cookieFile = __DIR__ . '/../cookies/ig-cookie.txt';
        file_put_contents($cookieFile, get_option('aiodl_instagram_cookies'));
    }

    public static function checkSettingsCache()
    {
        $cacheFile = __DIR__ . '/../settings.json';
        $cache = json_decode(file_get_contents($cacheFile), true);
        if ((time() - filemtime($cacheFile)) >= 86400 || empty($cache['siteUrl'])) {
            $option['siteUrl'] = get_site_url();
            $option['hideDash'] = get_site_url('aiodl_hide_dash');
            $option['showMp3'] = get_site_url('aiodl_show_mp3');
            file_put_contents($cacheFile, json_encode($option));
        }
    }

    /**
     * Main All_in_One_Video_Downloader_Settings Instance
     *
     * Ensures only one instance of All_in_One_Video_Downloader_Settings is loaded or can be loaded.
     *
     * @param object $parent Object instance.
     * @return object All_in_One_Video_Downloader_Settings instance
     * @since 1.0.0
     * @static
     * @see All_in_One_Video_Downloader()
     */
    public static function instance($parent)
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self($parent);
        }
        return self::$_instance;
    } // End instance()

    /**
     * Cloning is forbidden.
     *
     * @since 1.0.0
     */
    public function __clone()
    {
        _doing_it_wrong(__FUNCTION__, esc_html(__('Cloning of All_in_One_Video_Downloader_API is forbidden.')), esc_attr($this->parent->_version));
    } // End __clone()

    /**
     * Unserializing instances of this class is forbidden.
     *
     * @since 1.0.0
     */
    public function __wakeup()
    {
        _doing_it_wrong(__FUNCTION__, esc_html(__('Unserializing instances of All_in_One_Video_Downloader_API is forbidden.')), esc_attr($this->parent->_version));
    } // End __wakeup()

}
